<?php

namespace App\Http\Controllers\API\v1;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Http\Requests\FeeInvoiceRequest;
use App\Http\Requests\FeeRequest;
use App\Http\Resources\FeeResource;
use App\Models\Fee;
use App\Models\FeeType;
use App\Models\Transaction;
use Carbon\Carbon;
use App\Models\Student;
use DB;

class FeeController extends Controller
{
    
    
    /**
     * fee_months
     *
     * @param  mixed $requests
     * @return void
     */
    public function fee_months(Request $request){
        $months = Fee::select('month')->groupBy('month')->orderBy('month', 'DESC')->get()->pluck('month');
        return response()->json(['months' => $months]);
    }
    
    /**
     * fee_types
     *
     * @param  mixed $request
     * @return void
     */
    public function fee_types(Request $request){
        $fee_types = FeeType::select('title')->orderBy('title', 'ASC')->get()->pluck('title');
        return response()->json(['fee_types' => $fee_types]);
    }
    
    /**
     * get
     *
     * @param  mixed $request
     * @param  mixed $month
     * @return void
     */
    public function get(Request $request, $month, $offset, $page_size){

        $class = $request->class ?? null;
        $section = $request->section ?? null;
        $campus = $request->campus ?? null;        
        $keywords = $request->keywords ?? null; 
        $date = Carbon::parse($month);

        $query = Fee::whereYear('month', '=', $date->year)->whereMonth('month', '=', $date->month);

        if($class){
            $query = $query->whereHas('student', function($q1) use($class){
                $q1->where('grade_id', $class);
            });
        }
        if($section){
            $query = $query->whereHas('student', function($q2) use($section){
                $q2->where('section_id', $section);
            });
        }
        if($campus){
            $query = $query->whereHas('student', function($q3) use($campus){
                $q3->where('campus_id', $campus);
            });
        }
        
        if($keywords){
            $query = $query->where(function($q) use ($keywords) {
                $q->whereHas('student', function($q) use($keywords){
                    $q->where(DB::raw("concat(first_name, ' ', last_name)"), 'LIKE', "%".$keywords."%")                            
                        ->orWhere('guardian_name', 'like', '%'. $keywords. '%');
                })->orWhere('serial',  $keywords);                
            });
        }
        $total = $query->count();    
        // $query = $query->get()->sortBy('total_balance');
        // $query = $query->orderByRaw('total_balance DESC');
        $fees = $query->skip($offset)->take($page_size)->get();

        return response()->json(['fees' => FeeResource::collection($fees), 'total' => $total]);

    }

    
    /**
     * create
     *
     * @param  mixed $request
     * @return void
     */
    public function create(FeeInvoiceRequest $request){

        if($request->type === 'BULK'){
            $students = Student::whereIn('grade_id', $request->classes)->whereIn('campus_id', $request->campuses)->get();            
        }

        if($request->type === 'INDIVIDUAL'){
            $students = Student::whereIn('id', $request->student_ids)->get();
        }

        foreach($students as $student){
            $fee = new Fee();
            $fee->student_id = $student->id;            
            $fee->month = $request->date('month');
            $fee->due_date = $request->date('due_date');
            $fee->status = $request->status;
            $fee->notes = $request->notes;
                        
            $total_charges = [];
            foreach($student->fees as $my_fee){
                if(in_array($my_fee['type'], $request->fee_types)){
                    $total_charges[$my_fee['type']] = $my_fee['amount'];
                }
            }       
            $fee->charges = $total_charges;
            $fee->save();
            $id_prefix = date('ymd').'-';
            $fee->serial = $id_prefix.$fee->id;
            $fee->save();
        }        
        $months = Fee::select('month')->groupBy('month')->orderBy('month', 'DESC')->get()->pluck('month');
        return response()->json(['months' => $months]);
    }



    public function update(FeeRequest $request, $id){
        $fee = Fee::find($id);
        $fee->month = $request->date('month');
        $fee->due_date = $request->date('due_date');        
        $fee->status = $request->status;        
        $charges = [];
        $balances = [];
        $amount = $request->total_amount;
        $all_charges = 0;
        foreach($request->charges as $student_fee){            
            $charges[$student_fee['type']] = $student_fee['amount'];
            $all_charges += floatval($student_fee['amount']);
        }
        
        $fee->charges = $charges;
        $fee->balance = floatval($all_charges) - floatval($request->total_amount);
        $fee->save();        


        if($request->status != 'PENDING') {
            
            $transaction = [
                'transactable_id'       => $fee->id,
                'transactable_type'     => 'App\Models\Fee',
                'user_id'               => auth()->user()->id,
                'ref'                   => $request->ref,
                'amount'                => $amount,
                'status'                => 'COMPLETED',
                'type'                  => 'IN'
            ];

            Transaction::updateOrCreate(['transactable_id'  => $fee->id, 'transactable_type' => 'App\Models\Fee'], $transaction);
        }

        return response()->json(['fee' => new FeeResource($fee)]);
    }
    
}
