<?php

namespace App\Http\Controllers\API\v1;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Student;
use App\Http\Resources\StudentResource;
use App\Http\Resources\StudentInfoResource;
use App\Http\Requests\StudentRequest;
use App\Http\Requests\StudentImportRequest;
use App\Http\Requests\FeePlanRequest;
use App\Services\MediaService;
use App\Imports\StudentImport;
use App\Models\SourceType;
use App\Models\FeeType;
use DB;
use Log;
use Excel;

class StudentController extends Controller
{    
    /**
     * get
     *
     * @param  mixed $request
     * @return void
     */
    public function get(Request $request, $offset, $page_size){


        $class = $request->class ?? null;
        $section = $request->section ?? null;
        $campus = $request->campus ?? null;
        $gender = $request->gender ?? null;
        $keywords = $request->keywords ?? null;        
        $query = Student::query();
        if($class){
            $query = $query->where('grade_id', $class);
        }
        if($section){
            $query = $query->where('section_id', $section);
        }
        if($campus){
            $query = $query->where('campus_id', $campus);
        }
        if($gender){
            $query = $query->where('gender', $gender);
        }

        if($keywords){
            $query = $query->where(function($q) use ($keywords) {
                $q->where(DB::raw("concat(first_name, ' ', last_name)"), 'LIKE', "%".$keywords."%")
                    ->orWhere('guardian_name', 'like', '%'. $keywords. '%')
                    ->orWhere('code', 'like', '%'. $keywords. '%')
                    ->orWhere('contacts->primary', 'like', '%'.$keywords.'%')
                    ->orWhere('cnic', 'like', '%'. $keywords. '%');                    
            });
        }
        $total = $query->count();       
        $students = $query->skip($offset)->take($page_size)->get();
        return response()->json(['students' => StudentResource::collection($students), 'total' => $total]);
    }

    
    /**
     * search
     *
     * @param  mixed $request
     * @return void
     */
    public function search(Request $request){
        $classes = $request->classes ?? [];
        $campuses = $request->campuses ?? [];
        
        $students = Student::whereIn('grade_id', $classes)->whereIn('campus_id', $campuses)->get();
        return response()->json(['students' => StudentInfoResource::collection($students)]);
    }

    
    /**
     * create
     *
     * @param  mixed $request
     * @return void
     */
    public function create(StudentRequest $request){
        $student = new Student();
        $student->number = $request->number;
        $student->code = self::_get_student_code();
        $student->address = $request->address;        
        $student->first_name = $request->first_name;        
        $student->last_name = $request->last_name;        
        $student->email = $request->email;   
        $student->gender = $request->gender;
        $student->cnic = $request->cnic;  
        $student->guardian_name = $request->guardian_name;        
        $student->contacts = ['primary' => $request->phone ?? "", 'secondary' => $request->secondary_phone ?? ""];                
        $student->admission_date = $request->date('admission_date');
        $student->dob = $request->date('dob');     
        $student->campus_id = $request->campus_id;
        $student->grade_id = $request->grade_id;        
        $student->section_id = $request->section_id;        
        $student->save();

        $image_path = MediaService::get_avatar_path($request);        
        if($image_path){
            $image_path['mediable_id']     = $student->id;
            $image_path['mediable_type']   = SourceType::STUDENT;
            MediaService::create($image_path);
        }

        $documents = MediaService::get_documents_path($request);
        if(count($documents) > 0){
            foreach($documents as $doc){
                $doc['mediable_id'] = $student->id;
                $doc['mediable_type'] = SourceType::STUDENT;
                MediaService::create($doc);
            }
        }
        
        return response()->json([ 'student' => new StudentResource($student) ]);
    }
    
    /**
     * update
     *
     * @param  mixed $request
     * @param  mixed $id
     * @return void
     */
    public function update(StudentRequest $request, $id){

        $student = Student::find($id);
        $student->number = $request->number;        
        $student->address = $request->address;        
        $student->first_name = $request->first_name;        
        $student->last_name = $request->last_name;        
        $student->email = $request->email;   
        $student->gender = $request->gender;
        $student->cnic = $request->cnic;  
        $student->guardian_name = $request->guardian_name;        
        $student->contacts = ['primary' => $request->phone ?? "", 'secondary' => $request->secondary_phone ?? ""];               
        $student->admission_date = $request->date('admission_date');
        $student->dob = $request->date('dob');     
        $student->campus_id = $request->campus_id;
        $student->grade_id = $request->grade_id;        
        $student->section_id = $request->section_id;        
                
        
        if($request->has_avatar == "false"){
            MediaService::remove($student->id, SourceType::STUDENT, 'AVATAR');
        }
        $student->save();     

        $image_path = MediaService::get_avatar_path($request);        
        if($image_path){
            $image_path['mediable_id']     = $student->id;
            $image_path['mediable_type']   = SourceType::STUDENT;
            MediaService::create($image_path);
        }

        $documents = MediaService::get_documents_path($request);
        if(count($documents) > 0){
            foreach($documents as $doc){
                $doc['mediable_id'] = $student->id;
                $doc['mediable_type'] = SourceType::STUDENT;
                MediaService::create($doc);
            }
        }
        
        return response()->json([ 'student' => new StudentResource($student) ]);
    }

    public function feeplan(FeePlanRequest $request, $id){
        $student = Student::find($id);
        $student->fees = $request->fees;
        $student->save();

        foreach($request->fees as $fee){
            FeeType::updateOrCreate(['title' => ucwords($fee['type']) ], ['title' => ucwords($fee['type']) ] );
        } 

        return response()->json([ 'student' => new StudentResource($student) ]);

    }


    public  function _get_student_code(){
                
        $id_prefix = 'AHE-'.date('y');
        $old_id = Student::where('code', 'like', '%'.$id_prefix.'%')->orderBy('created_at', 'DESC')->limit(1)->first();
        if($old_id){
            $previous_id = substr($old_id->code, -4);
            $new_id = ++$previous_id;      
            $new_id = sprintf("%04d", $new_id);
        }else{
            $new_id = "0001";      
        } 

        return $id_prefix.$new_id;
        
    }   
    
    /**
     * import
     *
     * @param  mixed $request
     * @return void
     */
    public function import(StudentImportRequest $request){                   
        try{
            $test = Excel::import(new StudentImport($request->grade_id, $request->section_id, $request->campus_id),$request->file('file'));
            return response()->json(['message' => 'Students Imported Successfully'], 200);                               
        }catch(\Exception $e){
            return response()->json(['error' => 'Action Failed!', 'file' => $e->getFile(), 'line' => $e->getLine(), 'message' => $e->getMessage()], 409);
        }        
    }


    public function delete(Request $request, $id){
        Student::delete($id);        
        return response()->json([
            'id' => intval($id)
        ]);
    }
}
